/*
 * Decompiled with CFR 0.152.
 */
package com.l2jserver.gameserver.instancemanager;

import com.l2jserver.commons.database.ConnectionFactory;
import com.l2jserver.commons.util.Rnd;
import com.l2jserver.gameserver.ThreadPoolManager;
import com.l2jserver.gameserver.config.Configuration;
import com.l2jserver.gameserver.datatables.SpawnTable;
import com.l2jserver.gameserver.instancemanager.DayNightSpawnManager;
import com.l2jserver.gameserver.model.L2Spawn;
import com.l2jserver.gameserver.model.StatsSet;
import com.l2jserver.gameserver.model.actor.instance.L2RaidBossInstance;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.Calendar;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RaidBossSpawnManager {
    private static final Logger LOG = LoggerFactory.getLogger(RaidBossSpawnManager.class);
    private static final int EILHALDER_VON_HELLMANN = 25328;
    protected static final Map<Integer, L2RaidBossInstance> _bosses = new ConcurrentHashMap<Integer, L2RaidBossInstance>();
    protected static final Map<Integer, L2Spawn> _spawns = new ConcurrentHashMap<Integer, L2Spawn>();
    protected static final Map<Integer, StatsSet> _storedInfo = new ConcurrentHashMap<Integer, StatsSet>();
    protected static final Map<Integer, ScheduledFuture<?>> _schedules = new ConcurrentHashMap();

    protected RaidBossSpawnManager() {
        this.load();
    }

    public void load() {
        LOG.info("Spawning raid bosses...");
        _bosses.clear();
        _spawns.clear();
        _storedInfo.clear();
        _schedules.clear();
        try (Connection con = ConnectionFactory.getInstance().getConnection();
             Statement s = con.createStatement();
             ResultSet rs = s.executeQuery("SELECT * FROM raidboss_spawnlist ORDER BY boss_id");){
            while (rs.next()) {
                L2Spawn spawnDat = new L2Spawn(rs.getInt("boss_id"));
                spawnDat.setX(rs.getInt("loc_x"));
                spawnDat.setY(rs.getInt("loc_y"));
                spawnDat.setZ(rs.getInt("loc_z"));
                spawnDat.setAmount(rs.getInt("amount"));
                spawnDat.setHeading(rs.getInt("heading"));
                spawnDat.setRespawnDelay(rs.getInt("respawn_delay"), rs.getInt("respawn_random"));
                this.addNewSpawn(spawnDat, rs.getLong("respawn_time"), rs.getDouble("currentHP"), rs.getDouble("currentMP"), false);
            }
            LOG.info("Loaded {} bosses.", (Object)_bosses.size());
            LOG.info("Scheduled {} boss instances.", (Object)_schedules.size());
        }
        catch (Exception ex) {
            LOG.warn("There has been an error while initializing raid boss spawn manager!", ex);
        }
    }

    public void updateStatus(L2RaidBossInstance boss, boolean isBossDead) {
        StatsSet info = _storedInfo.get(boss.getId());
        if (info == null) {
            return;
        }
        if (isBossDead) {
            boss.setRaidStatus(StatusEnum.DEAD);
            int respawnMinDelay = (int)((double)boss.getSpawn().getRespawnMinDelay() * Configuration.npc().getRaidMinRespawnMultiplier());
            int respawnMaxDelay = (int)((double)boss.getSpawn().getRespawnMaxDelay() * Configuration.npc().getRaidMaxRespawnMultiplier());
            int respawnDelay = Rnd.get((int)respawnMinDelay, (int)respawnMaxDelay);
            long respawnTime = Calendar.getInstance().getTimeInMillis() + (long)respawnDelay;
            info.set("currentHP", boss.getMaxHp());
            info.set("currentMP", boss.getMaxMp());
            info.set("respawnTime", respawnTime);
            if (!(_schedules.containsKey(boss.getId()) || respawnMinDelay <= 0 && respawnMaxDelay <= 0)) {
                Calendar time = Calendar.getInstance();
                time.setTimeInMillis(respawnTime);
                LOG.info("Updated {} respawn time to {}.", (Object)boss.getName(), (Object)time.getTime());
                _schedules.put(boss.getId(), ThreadPoolManager.getInstance().scheduleGeneral(new SpawnSchedule(boss.getId()), respawnDelay));
                this.updateDb();
            }
        } else {
            boss.setRaidStatus(StatusEnum.ALIVE);
            info.set("currentHP", boss.getCurrentHp());
            info.set("currentMP", boss.getCurrentMp());
            info.set("respawnTime", 0L);
        }
        _storedInfo.put(boss.getId(), info);
    }

    public void addNewSpawn(L2Spawn spawnDat, long respawnTime, double currentHP, double currentMP, boolean storeInDb) {
        if (spawnDat == null) {
            return;
        }
        if (_spawns.containsKey(spawnDat.getId())) {
            return;
        }
        int bossId = spawnDat.getId();
        long time = Calendar.getInstance().getTimeInMillis();
        SpawnTable.getInstance().addNewSpawn(spawnDat, false);
        if (respawnTime == 0L || time > respawnTime) {
            L2RaidBossInstance raidBoss = bossId == 25328 ? DayNightSpawnManager.getInstance().handleBoss(spawnDat) : (L2RaidBossInstance)spawnDat.doSpawn();
            if (raidBoss != null) {
                raidBoss.setCurrentHp(currentHP);
                raidBoss.setCurrentMp(currentMP);
                raidBoss.setRaidStatus(StatusEnum.ALIVE);
                _bosses.put(bossId, raidBoss);
                StatsSet info = new StatsSet();
                info.set("currentHP", currentHP);
                info.set("currentMP", currentMP);
                info.set("respawnTime", 0L);
                _storedInfo.put(bossId, info);
            }
        } else {
            long spawnTime = respawnTime - Calendar.getInstance().getTimeInMillis();
            _schedules.put(bossId, ThreadPoolManager.getInstance().scheduleGeneral(new SpawnSchedule(bossId), spawnTime));
        }
        _spawns.put(bossId, spawnDat);
        if (storeInDb) {
            try (Connection con = ConnectionFactory.getInstance().getConnection();
                 PreparedStatement ps = con.prepareStatement("INSERT INTO raidboss_spawnlist (boss_id,amount,loc_x,loc_y,loc_z,heading,respawn_time,currentHp,currentMp) VALUES(?,?,?,?,?,?,?,?,?)");){
                ps.setInt(1, spawnDat.getId());
                ps.setInt(2, spawnDat.getAmount());
                ps.setInt(3, spawnDat.getX());
                ps.setInt(4, spawnDat.getY());
                ps.setInt(5, spawnDat.getZ());
                ps.setInt(6, spawnDat.getHeading());
                ps.setLong(7, respawnTime);
                ps.setDouble(8, currentHP);
                ps.setDouble(9, currentMP);
                ps.execute();
            }
            catch (Exception ex) {
                LOG.warn("Could not store raid boss Id {} in the DB!", (Object)bossId, (Object)ex);
            }
        }
    }

    public void deleteSpawn(L2Spawn spawnDat, boolean updateDb) {
        if (spawnDat == null) {
            return;
        }
        int bossId = spawnDat.getId();
        if (!_spawns.containsKey(bossId)) {
            return;
        }
        SpawnTable.getInstance().deleteSpawn(spawnDat, false);
        _spawns.remove(bossId);
        _bosses.remove(bossId);
        ScheduledFuture<?> f = _schedules.remove(bossId);
        if (f != null) {
            f.cancel(true);
        }
        _storedInfo.remove(bossId);
        if (updateDb) {
            try (Connection con = ConnectionFactory.getInstance().getConnection();
                 PreparedStatement ps = con.prepareStatement("DELETE FROM raidboss_spawnlist WHERE boss_id=?");){
                ps.setInt(1, bossId);
                ps.execute();
            }
            catch (Exception ex) {
                LOG.warn("Could not remove raid boss Id {} from DB!", (Object)bossId, (Object)ex);
            }
        }
    }

    private void updateDb() {
        try (Connection con = ConnectionFactory.getInstance().getConnection();
             PreparedStatement ps = con.prepareStatement("UPDATE raidboss_spawnlist SET respawn_time = ?, currentHP = ?, currentMP = ? WHERE boss_id = ?");){
            for (Integer bossId : _storedInfo.keySet()) {
                StatsSet info;
                L2RaidBossInstance boss;
                if (bossId == null || (boss = _bosses.get(bossId)) == null) continue;
                if (boss.getRaidStatus().equals((Object)StatusEnum.ALIVE)) {
                    this.updateStatus(boss, false);
                }
                if ((info = _storedInfo.get(bossId)) == null) continue;
                try {
                    ps.setLong(1, info.getLong("respawnTime"));
                    ps.setDouble(2, info.getDouble("currentHP"));
                    ps.setDouble(3, info.getDouble("currentMP"));
                    ps.setInt(4, bossId);
                    ps.executeUpdate();
                    ps.clearParameters();
                }
                catch (Exception ex) {
                    LOG.warn("Could not update raid boss spawn list table!", ex);
                }
            }
        }
        catch (Exception ex) {
            LOG.warn("SQL error while updating Raid Boss spawn to database!", ex);
        }
    }

    public String[] getAllRaidBossStatus() {
        String[] msg = new String[Math.max(_bosses.size(), 1)];
        if (_bosses.isEmpty()) {
            msg[0] = "None";
            return msg;
        }
        int index = 0;
        for (L2RaidBossInstance boss : _bosses.values()) {
            msg[index++] = boss.getName() + ": " + boss.getRaidStatus().name();
        }
        return msg;
    }

    public String getRaidBossStatus(int bossId) {
        String msg = "RaidBoss Status..." + Configuration.EOL;
        if (_bosses == null) {
            msg = msg + "None";
            return msg;
        }
        if (_bosses.containsKey(bossId)) {
            L2RaidBossInstance boss = _bosses.get(bossId);
            msg = msg + boss.getName() + ": " + boss.getRaidStatus().name();
        }
        return msg;
    }

    public StatusEnum getRaidBossStatusId(int bossId) {
        if (_bosses.containsKey(bossId)) {
            return _bosses.get(bossId).getRaidStatus();
        }
        if (_schedules.containsKey(bossId)) {
            return StatusEnum.DEAD;
        }
        return StatusEnum.UNDEFINED;
    }

    public void notifySpawnNightBoss(L2RaidBossInstance raidBoss) {
        StatsSet info = new StatsSet();
        info.set("currentHP", raidBoss.getCurrentHp());
        info.set("currentMP", raidBoss.getCurrentMp());
        info.set("respawnTime", 0L);
        raidBoss.setRaidStatus(StatusEnum.ALIVE);
        _storedInfo.put(raidBoss.getId(), info);
        LOG.info("Spawning Night Raid Boss {}.", (Object)raidBoss.getName());
        _bosses.put(raidBoss.getId(), raidBoss);
    }

    public boolean isDefined(int bossId) {
        return _spawns.containsKey(bossId);
    }

    public Map<Integer, L2RaidBossInstance> getBosses() {
        return _bosses;
    }

    public Map<Integer, L2Spawn> getSpawns() {
        return _spawns;
    }

    public Map<Integer, StatsSet> getStoredInfo() {
        return _storedInfo;
    }

    public void cleanUp() {
        this.updateDb();
        _bosses.clear();
        if (_schedules != null) {
            for (Integer bossId : _schedules.keySet()) {
                ScheduledFuture<?> f = _schedules.get(bossId);
                f.cancel(true);
            }
            _schedules.clear();
        }
        _storedInfo.clear();
        _spawns.clear();
    }

    public static RaidBossSpawnManager getInstance() {
        return SingletonHolder.INSTANCE;
    }

    public static enum StatusEnum {
        ALIVE,
        DEAD,
        UNDEFINED;

    }

    private static class SpawnSchedule
    implements Runnable {
        private static final Logger LOG = LoggerFactory.getLogger(SpawnSchedule.class);
        private final int bossId;

        public SpawnSchedule(int npcId) {
            this.bossId = npcId;
        }

        @Override
        public void run() {
            L2RaidBossInstance raidBoss = this.bossId == 25328 ? DayNightSpawnManager.getInstance().handleBoss(_spawns.get(this.bossId)) : (L2RaidBossInstance)_spawns.get(this.bossId).doSpawn();
            if (raidBoss != null) {
                raidBoss.setRaidStatus(StatusEnum.ALIVE);
                StatsSet info = new StatsSet();
                info.set("currentHP", raidBoss.getCurrentHp());
                info.set("currentMP", raidBoss.getCurrentMp());
                info.set("respawnTime", 0L);
                _storedInfo.put(this.bossId, info);
                LOG.info("Spawning Raid Boss {}.", (Object)raidBoss.getName());
                _bosses.put(this.bossId, raidBoss);
            }
            _schedules.remove(this.bossId);
        }
    }

    private static class SingletonHolder {
        protected static final RaidBossSpawnManager INSTANCE = new RaidBossSpawnManager();

        private SingletonHolder() {
        }
    }
}

